MODULE logging

  ! see also use_logging.h
  ! macro LOGBUF accumulates log output into logging_buffer

  IMPLICIT NONE
  SAVE
  PRIVATE

  INTERFACE  ! Explicit interfaces for plugins

     ! Plugin that typically prints all lines in the loggin buffer 'buf' and prepends tags (log level, timestamp, ...)
     SUBROUTINE plugin(lev, tag, buf)
       INTEGER, INTENT(IN) :: lev
       CHARACTER(*), INTENT(IN) :: tag, buf(:)
     END SUBROUTINE plugin

     ! Plugin that writes into string 'line' information about the gridpoint of index 'index'
     SUBROUTINE plugin_log_gridpoint(index, line)
       INTEGER, INTENT(IN) :: index ! index of gridpoint
       CHARACTER(*), INTENT(OUT) :: line
     END SUBROUTINE plugin_log_gridpoint

  END INTERFACE


  ! This module provides a default implementations of plugins but the top-level driver is welcome to override them.
  ! Note F2003/F2008: pgfortran (F2003) accepts to initialize pointers only to NULL()
  ! => plugins are initialzed to NULL() and set to default values in flush_log and log_gridpoint
#ifndef XCODEML
  ! Note F2003/F2008: XCodeML cannot parse procedure pointers
  PROCEDURE(plugin), POINTER, PUBLIC :: flush_plugin => NULL()
  PROCEDURE(plugin_log_gridpoint), POINTER, PUBLIC :: log_gridpoint_plugin => NULL()
#endif

  INTEGER, PARAMETER :: linesize=10000, logging_bufsize=100
  CHARACTER(linesize) :: logging_buf(logging_bufsize)

  INTEGER :: logging_lineno=0

  ! messages with a log level > max_log_level are not printed
  INTEGER, PARAMETER, PUBLIC :: log_level_fatal=1, log_level_error=2, log_level_warn=3, log_level_info=4, log_level_dbg=5
  INTEGER, PUBLIC :: max_log_level = log_level_info

  PUBLIC :: logging_buf, logging_bufsize, logging_lineno, flush_log, log_gridpoint, &
       missing_plugin

CONTAINS

  SUBROUTINE missing_plugin(name, mod)
    CHARACTER(*), INTENT(IN) :: name, mod
    WRITE(logging_buf(1),*) 'WARNING : plugin ', name, ' not provided by the driver program'
    WRITE(logging_buf(2),*) '        see ', mod
    logging_lineno=2
    CALL flush_log(log_level_warn, 'missing_plugin')
  END SUBROUTINE missing_plugin

  SUBROUTINE flush_log(lev,tag)
    INTEGER, INTENT(IN) :: lev
    CHARACTER(*), INTENT(IN) :: tag
#ifndef XCODEML
    IF(.NOT.ASSOCIATED(flush_plugin)) THEN
       flush_plugin => default_flush_plugin
       WRITE(*,*) 'WARNING : plugin flush_plugin not provided by the driver program'
       WRITE(*,*) '        see logging.F90'
    END IF
    IF(logging_lineno>0 .AND. lev<=max_log_level) CALL flush_plugin(lev, TRIM(tag), logging_buf(1:logging_lineno))
    logging_lineno=0
#endif
  END SUBROUTINE flush_log

  SUBROUTINE default_flush_plugin(lev, tag, buf)
    INTEGER, INTENT(IN) :: lev
    CHARACTER(*), INTENT(IN) :: tag, buf(:)
    CHARACTER(3), DIMENSION(log_level_dbg) :: dbtag = (/ 'FAT', 'ERR', 'WRN', 'INF', 'DBG' /)
    CHARACTER(100) :: prefix
    INTEGER :: i
    WRITE(prefix,*) '[', dbtag(lev), ' ', tag, ']'
    DO i=1, SIZE(buf)
       WRITE(*,*) TRIM(prefix), TRIM(buf(i))
    END DO
  END SUBROUTINE default_flush_plugin

  SUBROUTINE log_gridpoint(index)
    INTEGER, INTENT(IN) :: index
    logging_lineno = logging_lineno+1
#ifndef XCODEML
    IF(.NOT.ASSOCIATED(log_gridpoint_plugin)) THEN
       log_gridpoint_plugin => default_log_gridpoint
       WRITE(*,*) 'WARNING : plugin log_gridpoint_plugin not provided by the driver program'
       WRITE(*,*) '        see logging.F90'
    END IF
    CALL log_gridpoint_plugin(index, logging_buf(logging_lineno))
#endif
  END SUBROUTINE log_gridpoint

  SUBROUTINE default_log_gridpoint(index, line)
    INTEGER, INTENT(IN) :: index ! index of gridpoint
    CHARACTER(*), INTENT(OUT) :: line
    line=' '
  END SUBROUTINE default_log_gridpoint

END MODULE logging
