#!/bin/bash

#---------------------- Arguments ------------------

function fullpath() {
    mkdir -p $(dirname "$1")
    echo "$(cd "$(dirname "$1")"; pwd)/$(basename "$1")"
}

function readargs_ce0l() {
    MYDIR=$(fullpath $2)
    MYSTORE=$(fullpath $3)
    ANNEEREF=$4
    mkdir -p $MYDIR
    mkdir -p $MYSTORE
    ln -sf $MYSTORE $MYDIR/store 
    echo "ce0l : MYDIR=$MYDIR MYSTORE=$MYSTORE"
}

function readargs_first() {
    STORE0=$(fullpath $2)
    MYDIR=$(fullpath $3)
    MYSTORE=$(fullpath $4)
    ANNEEREF=$5
    mkdir -p $MYDIR
    mkdir -p $MYSTORE
    ln -sf $STORE0 $MYDIR/store0 
    ln -sf $MYSTORE $MYDIR/store 
}

function readargs_next() {
    STORE0=$(fullpath $2)
    STOREPREV=$(fullpath $3)
    MYDIR=$(fullpath $4)
    MYSTORE=$(fullpath $5)
    ANNEEREF=$6
    mkdir -p $MYDIR
    mkdir -p $MYSTORE
    ln -sf $STORE0 $MYDIR/store0 
    ln -sf $STOREPREV $MYDIR/store_prev 
    ln -sf $MYSTORE $MYDIR/store 
}

function check_env()
{
    if [ "$ARCH" == "" ] ; then
	echo 'You must define environment variables ARCH and ESM_ROOT'
	exit 1
    fi
    if [ "$ESM_ROOT" == "" ] ; then
	echo 'You must define environment variables ARCH and ESM_ROOT'
	exit 1
    fi
    XIOS_SERVER="${ESM_ROOT}/XIOS/bin/xios_server.exe"
    if [ ! -f "${XIOS_SERVER}" ]; then
	echo "${XIOS_SERVER} not found, check that ESM_ROOT=$ESM_ROOT points to the correct location"
    fi
    
}

#---------------- Keyword substitution -------------

function def() {
    echo "s/$1/$2/g"
}

function sed_ce0l() {
    def USING_SERVER true
    def CE0L yes
    def READ_START no
    def IFLAG_PHYS $IFLAG_PHYS0
    def ETAT0 $ETAT0
}

function sed_first() {
    def USING_SERVER true
    def CE0L no
    def READ_START yes
    def IFLAG_PHYS 1
    def ETAT0 $ETAT0
}

function sed_next() {
    def USING_SERVER true
    def CE0L no
    def READ_START yes
    def IFLAG_PHYS 1
    def ETAT0 start_file
}

function sed_all() {
    def READ_CLIMOZ $READ_CLIMOZ
    def KEY_VEGET $VEGET
    def SOLARLONG0 $SOLARLONG0
    def ANNEEREF $ANNEEREF
}

#------------ Batch jobs : machine-independent prologues and epilogues  --------------

function prologue_all() {
cat <<EOF 
cd ${MYDIR}
rm -rf gcm.log *.nc xios_*.err xios_*.out gcm.log logs
ln -s dynamico.0000.out gcm.log

symlink() { if [ -e \$1 ] ; then ln -s \$1 \$2 ; fi ; }

for x in \$(ls ../FILES_LMDZ/*.nc) ; do ln -s \$x . ; done

cat <<END  > killme.sh
#!/bin/bash
ccc_mdel \${BRIDGE_MSUB_JOBID}
END
chmod a+x killme.sh

ulimit -s unlimited

EOF
cat $ESM_ROOT/ICOSA_LMDZ/arch.env
}

function prologue_ce0l() {
echo
}

function prologue_first() {
cat <<EOF 
symlink store0/limit.nc .
symlink store0/climoz_LMDZ.nc .
symlink store0/startphy.nc .
EOF
}

function prologue_next() {
cat <<EOF 
symlink store0/limit.nc .
symlink store0/climoz_LMDZ.nc .
symlink store_prev/restartphy.nc startphy.nc
symlink store_prev/restart.nc start.nc
EOF
}

function epilogue_ce0l(){
    cat <<EOF
# save useful files
mv limit.nc startphy.nc climoz_LMDZ.nc store/
EOF
}

function epilogue_first(){
    cat <<EOF
# save useful files
mv restartphy.nc restart.nc store/
EOF
}

function epilogue_next(){
    cat <<EOF
# save useful files
mv restartphy.nc restart.nc store/
EOF
}

function epilogue_all() {
    cat <<EOF
mkdir logs
cp *.def *.xml job.sh logs/
mv apbp.nc Ai.nc *.err *.out gcm.log logs/
tar czf logs.tgz logs
mv hist*.nc logs.tgz store/
# cleanup input files
rm -f *.nc
EOF
}

#------------ Batch jobs : machine-dependent main script : X64_CURIE  --------------

function batch_X64_CURIE_ce0l() {
batch_X64_CURIE ce0l $WALLTIME_CE0L
}

function batch_X64_CURIE_first() {
batch_X64_CURIE first $WALLTIME_JOB
}

function batch_X64_CURIE_next() {
batch_X64_CURIE next $WALLTIME_JOB
}

function batch_X64_CURIE() { # $1 = first or next $2 = walltime
NB_MPI_TOTAL=$(( $NB_MPI_DYNAMICO + $NB_MPI_XIOS ))
cat <<EOF
#!/bin/bash
## Request name
#MSUB -r ${EXPERIMENT}_$1_mpi
#MSUB -q standard 
#MSUB -A gen7548
#MSUB -x
## Number of tasks (=MPI processes) to use
#MSUB -n $NB_MPI_TOTAL
## Elapsed time limit in seconds
#MSUB -T $2
## Quality of Service required (long [3 days], normal [1 day], test [30 min])
#MSUB -Q normal

$( prologue_all )
$( prologue_$1 )
export OMP_NUM_THREADS=1
ccc_mprun -f mpmd.conf

$( epilogue_$1 )
$( epilogue_all )
EOF

cat <<EOF > $MYDIR/mpmd.conf
$NB_MPI_DYNAMICO ./gcm.sh
$NB_MPI_XIOS $XIOS_SERVER
EOF

cat <<EOF > $MYDIR/gcm.sh
#!/bin/bash
ME=\${BRIDGE_MPRUN_RANKID}
GCM=\$(printf 'dynamico.%04d' \$ME)
date > \${GCM}.out
$ICOSA_LMDZ >> \${GCM}.out 2>\${GCM}.err
date >> \${GCM}.out
EOF
chmod a+x $MYDIR/gcm.sh
}

#------------ Batch jobs : machine-dependent main script : X64_IRENE  --------------

function batch_X64_IRENE_ce0l() {
batch_X64_IRENE ce0l $WALLTIME_CE0L
}

function batch_X64_IRENE_first() {
batch_X64_IRENE first $WALLTIME_JOB
}

function batch_X64_IRENE_next() {
batch_X64_IRENE next $WALLTIME_JOB
}

function batch_X64_IRENE() { # $1 = first or next $2 = walltime
NB_MPI_TOTAL=$(( $NB_MPI_DYNAMICO + $NB_MPI_XIOS ))
cat <<EOF
#!/bin/bash
## Request name
#MSUB -r ${EXPERIMENT}_$1_mpi
#MSUB -q skylake
#MSUB -A gch0316
#MSUB -m work,scratch
#MSUB -x
## Number of tasks (=MPI processes) to use
#MSUB -n $NB_MPI_TOTAL
## Elapsed time limit in seconds
#MSUB -T $2

$( prologue_all )
module load ipm
export IPM_REPORT=full
$( prologue_$1 )
export OMP_NUM_THREADS=1
ccc_mprun -f mpmd.conf
awk '/IPMv2/{y=1;next}y' gcm.log
grep MAIN gcm.log
$( epilogue_$1 )
$( epilogue_all )
EOF

cat <<EOF > $MYDIR/mpmd.conf
$NB_MPI_DYNAMICO ./gcm.sh
$NB_MPI_XIOS $XIOS_SERVER
EOF

cat <<EOF > $MYDIR/gcm.sh
#!/bin/bash

function run()
{
    hostname
    ldd $ICOSA_LMDZ
    $ICOSA_LMDZ
}

function run_print_each()
{
    GCM=\$(printf 'dynamico.%04d' \$ME)
    date > \${GCM}.out
    run >> \${GCM}.out 2>\${GCM}.err
    date >> \${GCM}.out
}

function run_print_master()
{
    if ((\$ME>0)) ; then
        run > /dev/null 2>&1
    else
        run_print_each
    fi
}

ME=\${BRIDGE_MPRUN_RANKID}

run_print_master
#run_print_each
EOF
chmod a+x $MYDIR/gcm.sh
}

#--------------------- Main --------------------

echo
echo
echo "Your command : $0 $*"
echo "Working directory : $PWD"

NB_MPI_DYNAMICO=160
NB_MPI_XIOS=16
NB_MPI_CE0L=16
source ./config.sh

check_env || exit 1

XIOS_SERVER=$(fullpath "$ESM_ROOT/XIOS/bin/xios_server.exe")
ICOSA_LMDZ=$(fullpath "$ESM_ROOT/ICOSA_LMDZ/bin/icosa_lmdz.exe")

TYPE=$1
readargs_$TYPE $* || exit

# avoid destroying xml and def files in case of mistake
chmod a-w *.def *.xml

MYSED=$(mktemp)
sed_$TYPE > $MYSED
sed_all >> $MYSED
#cat $MYSED
for FILE in *.def *.xml ; do
    sed -f $MYSED $FILE > $MYDIR/$FILE
done
rm -f $MYSED

chmod u+w *.def *.xml

cd $MYDIR
batch_${ARCH}_${TYPE} > job.sh

#ls -l $MYDIR
grep using_server iodef.xml
for KEY in create_etat0_limit read_start etat0 anneeref iflag_phys read_climoz VEGET solarlong0 ; do
    grep "\b$KEY\b" *.def | grep -v '#' # match whole words : \b means 'word boundary'
done
