MODULE physics_mod

  USE field_mod

  PRIVATE

  INTEGER, PARAMETER :: phys_none=0, phys_HS94=1, phys_DCMIP=2, phys_lmdz_generic=3, phys_LB2012=4, phys_external=5

  INTEGER :: phys_type
  TYPE(t_field),POINTER :: f_extra_physics_2D(:), f_extra_physics_3D(:)
  TYPE(t_field),POINTER :: f_dulon(:), f_dulat(:)
  TYPE(t_field),POINTER :: f_temp(:)

  CHARACTER(LEN=255) :: physics_type
!$OMP THREADPRIVATE(physics_type)

  PUBLIC :: physics, init_physics

CONTAINS

  SUBROUTINE init_physics
    USE mpipara
    USE etat0_mod
    USE icosa
    USE physics_interface_mod
    USE physics_dcmip_mod, ONLY : init_physics_dcmip=>init_physics
    USE etat0_venus_mod, ONLY : init_phys_venus=>init_physics
    USE physics_lmdz_generic_mod, ONLY : init_physics_lmdz_generic=>init_physics
    USE physics_external_mod, ONLY : init_physics_external=>init_physics
    IMPLICIT NONE

    physics_inout%dt_phys = dt*itau_physics
    physics_type='none'
    CALL getin("physics",physics_type)
    SELECT CASE(TRIM(physics_type))
    CASE ('none')
       IF(is_mpi_root) PRINT*,"NO PHYSICAL PACKAGE USED"
       phys_type = phys_none
    CASE ('held_suarez')
       phys_type = phys_HS94
    CASE ('Lebonnois2012')
       phys_type = phys_LB2012
       CALL init_phys_venus

    CASE ('phys_lmdz_generic')
       CALL init_physics_lmdz_generic
       phys_type=phys_lmdz_generic
    CASE ('phys_external')
       CALL init_physics_external
       phys_type=phys_external
    CASE ('dcmip')
       CALL allocate_field(f_dulon,field_t,type_real,llm, name='dulon')
       CALL allocate_field(f_dulat,field_t,type_real,llm, name='dulat')
       CALL allocate_field(f_temp,field_t,type_real,llm, name='temp')
       CALL init_pack_before ! Compute physics_inout%ngrid and offsets used by pack/unpack
       CALL init_physics_dcmip
       CALL init_pack_after ! Defines Ai, lon, lat in physics_inout
       phys_type = phys_DCMIP
    CASE DEFAULT
       IF(is_mpi_root) PRINT*, 'init_physics : Bad selector for variable physics <',&
            TRIM(physics_type), '> options are <none>, <held_suarez>, <Lebonnois2012>, <dcmip>', &
                                '<phys_lmdz_generic>, <phys_external>'
       STOP
    END SELECT

    IF(is_mpi_root) PRINT *, 'phys_type = ',phys_type
  END SUBROUTINE init_physics

  SUBROUTINE physics(it,f_phis, f_ps, f_theta_rhodz, f_ue, f_wflux, f_q)
    USE icosa
    USE physics_interface_mod
    USE physics_lmdz_generic_mod, ONLY : physics_lmdz_generic => physics
    USE physics_external_mod, ONLY : physics_external => physics
    USE physics_dcmip_mod, ONLY : write_physics_dcmip => write_physics
    USE etat0_heldsz_mod
    USE etat0_venus_mod, ONLY : phys_venus => physics
    IMPLICIT NONE
    INTEGER, INTENT(IN)   :: it
    TYPE(t_field),POINTER :: f_phis(:)
    TYPE(t_field),POINTER :: f_ps(:)
    TYPE(t_field),POINTER :: f_theta_rhodz(:)
    TYPE(t_field),POINTER :: f_ue(:)
    TYPE(t_field),POINTER :: f_wflux(:)
    TYPE(t_field),POINTER :: f_q(:)
    REAL(rstd),POINTER :: phis(:)
    REAL(rstd),POINTER :: ps(:)
    REAL(rstd),POINTER :: theta_rhodz(:,:)
    REAL(rstd),POINTER :: ue(:,:)
    REAL(rstd),POINTER :: q(:,:,:)

    LOGICAL:: firstcall,lastcall
    INTEGER :: ind
    TYPE(t_physics_inout) :: args

    IF(MOD(it,itau_physics)==0) THEN
    
       SELECT CASE(phys_type)
       CASE (phys_none)
          ! No physics, do nothing
       CASE(phys_HS94)
          CALL held_suarez(f_ps,f_theta_rhodz,f_ue) 
       CASE (phys_lmdz_generic)
         CALL physics_lmdz_generic(it ,f_phis, f_ps, f_theta_rhodz, f_ue, f_wflux, f_q)
       CASE (phys_external)
         CALL physics_external(it ,f_phis, f_ps, f_theta_rhodz, f_ue, f_wflux, f_q)
       CASE(phys_LB2012)
          CALL phys_venus(f_ps,f_theta_rhodz,f_ue) 
       CASE DEFAULT
          CALL physics_column(it, f_phis, f_ps, f_theta_rhodz, f_ue, f_q)
       END SELECT

       CALL transfert_request(f_theta_rhodz,req_i0)
       CALL transfert_request(f_ue,req_e0_vect)
       CALL transfert_request(f_q,req_i0)
    END IF

    IF (mod(it,itau_out)==0 ) THEN
       SELECT CASE(phys_type)
       CASE (phys_DCMIP)
          CALL write_physics_dcmip
       END SELECT
    END IF
    
  END SUBROUTINE physics

  SUBROUTINE physics_column(it, f_phis, f_ps, f_theta_rhodz, f_ue, f_q)
    USE icosa
    USE physics_interface_mod
    USE physics_dcmip_mod, ONLY : full_physics_dcmip => full_physics
    USE theta2theta_rhodz_mod
    USE mpipara
    IMPLICIT NONE
    TYPE(t_field),POINTER :: f_phis(:)
    TYPE(t_field),POINTER :: f_ps(:)
    TYPE(t_field),POINTER :: f_theta_rhodz(:)
    TYPE(t_field),POINTER :: f_ue(:)
    TYPE(t_field),POINTER :: f_q(:)
    REAL(rstd),POINTER :: phis(:)
    REAL(rstd),POINTER :: ps(:)
    REAL(rstd),POINTER :: temp(:,:)
    REAL(rstd),POINTER :: theta_rhodz(:,:)
    REAL(rstd),POINTER :: ue(:,:)
    REAL(rstd),POINTER :: dulon(:,:)
    REAL(rstd),POINTER :: dulat(:,:)
    REAL(rstd),POINTER :: q(:,:,:)
    INTEGER :: it, ind

    CALL theta_rhodz2temperature(f_ps,f_theta_rhodz,f_temp)

    DO ind=1,ndomain
       IF (.NOT. assigned_domain(ind)) CYCLE
       CALL swap_dimensions(ind)
       CALL swap_geometry(ind)
       phis=f_phis(ind)
       ps=f_ps(ind)
       temp=f_temp(ind)
       ue=f_ue(ind)
       q=f_q(ind)
       CALL pack_physics(pack_info(ind), phis, ps, temp, ue, q)
    END DO

    SELECT CASE(phys_type)
    CASE (phys_DCMIP)
       CALL full_physics_dcmip
    CASE DEFAULT
       IF(is_mpi_master) PRINT *,'Internal error : illegal value of phys_type', phys_type
       STOP
    END SELECT

    DO ind=1,ndomain
       IF (.NOT. assigned_domain(ind)) CYCLE
       CALL swap_dimensions(ind)
       CALL swap_geometry(ind)
       ps=f_ps(ind)
       temp=f_temp(ind)
       q=f_q(ind)
       dulon=f_dulon(ind)
       dulat=f_dulat(ind)
       CALL unpack_physics(pack_info(ind), ps, temp, q, dulon, dulat)
    END DO
    CALL temperature2theta_rhodz(f_ps,f_temp,f_theta_rhodz)

    ! Transfer dulon, dulat
    CALL transfert_request(f_dulon,req_i0)
    CALL transfert_request(f_dulat,req_i0)

    DO ind=1,ndomain
       IF (.NOT. assigned_domain(ind)) CYCLE
       CALL swap_dimensions(ind)
       CALL swap_geometry(ind)
       ue=f_ue(ind)
       dulon=f_dulon(ind)
       dulat=f_dulat(ind)
       CALL compute_update_velocity(dulon, dulat, ue)
    END DO

  END SUBROUTINE physics_column

  SUBROUTINE pack_physics(info, phis, ps, temp, ue, q)
    USE icosa
    USE wind_mod
    USE pression_mod
    USE theta2theta_rhodz_mod
    USE physics_interface_mod
    IMPLICIT NONE
    TYPE(t_pack_info) :: info
    REAL(rstd) :: phis(iim*jjm)
    REAL(rstd) :: ps(iim*jjm)
    REAL(rstd) :: temp(iim*jjm,llm)
    REAL(rstd) :: ue(3*iim*jjm,llm)
    REAL(rstd) :: q(iim*jjm,llm,nqtot)

    REAL(rstd) :: p(iim*jjm,llm+1)
    REAL(rstd) :: uc(iim*jjm,3,llm)
    REAL(rstd) :: ulon(iim*jjm,llm)
    REAL(rstd) :: ulat(iim*jjm,llm)

!$OMP BARRIER
    CALL compute_pression(ps,p,0)
!$OMP BARRIER
    CALL compute_wind_centered(ue,uc)
    CALL compute_wind_centered_lonlat_compound(uc, ulon, ulat)

    CALL pack_domain(info, phis, physics_inout%phis)
    CALL pack_domain(info, p, physics_inout%p)
    CALL pack_domain(info, Temp, physics_inout%Temp)
    CALL pack_domain(info, ulon, physics_inout%ulon)
    CALL pack_domain(info, ulat, physics_inout%ulat)
    CALL pack_domain(info, q, physics_inout%q)
  END SUBROUTINE pack_physics

  SUBROUTINE unpack_physics(info, ps,temp, q, dulon, dulat)
    USE icosa
    USE physics_interface_mod
    USE theta2theta_rhodz_mod
    IMPLICIT NONE
    TYPE(t_pack_info) :: info
    REAL(rstd) :: ps(iim*jjm)
    REAL(rstd) :: temp(iim*jjm,llm)
    REAL(rstd) :: q(iim*jjm,llm,nqtot)
    REAL(rstd) :: dulon(iim*jjm,llm)
    REAL(rstd) :: dulat(iim*jjm,llm)

    REAL(rstd) :: dq(iim*jjm,llm,nqtot)
    REAL(rstd) :: dTemp(iim*jjm,llm)
    CALL unpack_domain(info, dulon, physics_inout%dulon)
    CALL unpack_domain(info, dulat, physics_inout%dulat)
    CALL unpack_domain(info, dq, physics_inout%dq)
    CALL unpack_domain(info, Temp, physics_inout%Temp)
    CALL unpack_domain(info, dTemp, physics_inout%dTemp)
    q = q + physics_inout%dt_phys * dq
    Temp = Temp + physics_inout%dt_phys * dTemp
!    CALL compute_temperature2theta_rhodz(ps,Temp,theta_rhodz,0)
  END SUBROUTINE unpack_physics

  SUBROUTINE compute_update_velocity(dulon, dulat, ue)
    USE icosa
    USE physics_interface_mod
    USE wind_mod
    IMPLICIT NONE
    REAL(rstd) :: dulon(iim*jjm,llm)
    REAL(rstd) :: dulat(iim*jjm,llm)
    REAL(rstd) :: ue(3*iim*jjm,llm)
    REAL(rstd) :: duc(iim*jjm,3,llm)
    REAL(rstd) :: dt2, due
    INTEGER :: i,j,ij,l
    ! Reconstruct wind tendencies at edges and add to normal wind
    CALL compute_wind_centered_from_lonlat_compound(dulon,dulat,duc)
    dt2=.5*physics_inout%dt_phys
    DO l=1,llm
      DO j=jj_begin,jj_end
        DO i=ii_begin,ii_end
          ij=(j-1)*iim+i
          due = sum( (duc(ij,:,l) + duc(ij+t_right,:,l))*ep_e(ij+u_right,:) )
          ue(ij+u_right,l) = ue(ij+u_right,l) + dt2*due

          due = sum( (duc(ij,:,l) + duc(ij+t_lup,:,l))*ep_e(ij+u_lup,:) )
          ue(ij+u_lup,l)=ue(ij+u_lup,l) + dt2*due

          due = sum( (duc(ij,:,l) + duc(ij+t_ldown,:,l))*ep_e(ij+u_ldown,:) )
          ue(ij+u_ldown,l)=ue(ij+u_ldown,l) + dt2*due
        ENDDO
      ENDDO
    ENDDO
  END SUBROUTINE compute_update_velocity

END MODULE physics_mod
