MODULE geometry_mod

! Store informations concerning the local (to MPI/OpenMP core) geometry

  REAL,SAVE,ALLOCATABLE :: longitude(:) ! longitude of the cell (rad)
!$OMP THREADPRIVATE(longitude)

  REAL,SAVE,ALLOCATABLE :: latitude(:)! latitude of the cell (rad)
!$OMP THREADPRIVATE(latitude)

  REAL,SAVE,ALLOCATABLE :: longitude_deg(:) ! longitude of the cell (degree)
!$OMP THREADPRIVATE(longitude_deg)

  REAL,SAVE,ALLOCATABLE :: latitude_deg(:)! latitude of the cell (degree)
!$OMP THREADPRIVATE(latitude_deg)

  REAL,SAVE,ALLOCATABLE :: boundslon(:,:)  ! boundaries of the cell (rad)
!$OMP THREADPRIVATE(boundslon)

  REAL,SAVE,ALLOCATABLE :: boundslat(:,:) ! boundaries of the cell (rad)
!$OMP THREADPRIVATE(boundslat)

  REAL,SAVE,ALLOCATABLE :: dx(:)      ! resolution of longitude cell (valid only for 2D grid)
!$OMP THREADPRIVATE(dx)
  
  REAL,SAVE,ALLOCATABLE :: dy(:)      ! resolution of latitude cell (valid only for 2D grid)
!$OMP THREADPRIVATE(dy)

  REAL,SAVE,ALLOCATABLE :: cell_area(:)      ! area of the cell
!$OMP THREADPRIVATE(cell_area)


CONTAINS

  SUBROUTINE init_geometry(klon,longitude_,latitude_, &
                           boundslon_,boundslat_, &
                           cell_area_,dx_,dy_)
  USE mod_grid_phy_lmdz, ONLY: nvertex
  USE nrtype, ONLY : PI
  IMPLICIT NONE
    INTEGER,INTENT(IN) :: klon ! number of columns for this MPI/OpenMP domain
    REAL,INTENT(IN) :: longitude_(klon)
    REAL,INTENT(IN) :: latitude_(klon)
    REAL,INTENT(IN) :: boundslon_(klon,nvertex)
    REAL,INTENT(IN) :: boundslat_(klon,nvertex)
    REAL,INTENT(IN) :: cell_area_(klon)
    REAL,OPTIONAL,INTENT(IN) :: dx_(klon)
    REAL,OPTIONAL,INTENT(IN) :: dy_(klon)
    
    ALLOCATE(longitude(klon))
    ALLOCATE(latitude(klon))
    ALLOCATE(longitude_deg(klon))
    ALLOCATE(latitude_deg(klon))
    ALLOCATE(boundslon(klon,nvertex))
    ALLOCATE(boundslat(klon,nvertex))
    ALLOCATE(cell_area(klon))
    IF (PRESENT(dx_)) ALLOCATE(dx(klon))
    IF (PRESENT(dy_))ALLOCATE(dy(klon))
    
    longitude(:) = longitude_(:)
    latitude(:) = latitude_(:)
    longitude_deg(:) = longitude(:)*180./PI
    latitude_deg(:) = latitude(:)*180./PI
    boundslon(:,:) = boundslon_(:,:)
    boundslat(:,:) = boundslat_(:,:)
    cell_area(:) = cell_area_(:)
    IF (PRESENT(dx_)) dx(:) = dx_(:)
    IF (PRESENT(dy_)) dy(:) = dy_(:)
    
    ! Ehouarn debug
    write(*,*) "init_geometry: cell_area(1)=",cell_area(1)
  END SUBROUTINE init_geometry


END MODULE geometry_mod

