module FLOTT_GWD_rando_m

  implicit none

contains

  SUBROUTINE FLOTT_GWD_rando(DTIME, pp, tt, uu, vv, prec, zustr, zvstr, d_u, &
       d_v)

    ! Parametrization of the momentum flux deposition due to a discrete
    ! number of gravity waves. 
    ! Author: F. Lott
    ! July, 12th, 2012
    ! Gaussian distribution of the source, source is precipitation
    ! Reference: Lott (JGR, vol 118, page 8897, 2013)

    use dimphy, only: klon, klev
    use assert_m, only: assert

    include "YOMCST.h"
    include "clesphys.h"
    include "YOEGWD.h"

    ! 0. DECLARATIONS:

    ! 0.1 INPUTS
    REAL, intent(in)::DTIME ! Time step of the Physics
    REAL, intent(in):: pp(:, :) ! (KLON, KLEV) Pressure at full levels
    REAL, intent(in):: prec(:) ! (klon) Precipitation (kg/m^2/s) 
    REAL, intent(in):: TT(:, :) ! (KLON, KLEV) Temp at full levels 
    REAL, intent(in):: UU(:, :) ! (KLON, KLEV) Zonal wind at full levels
    REAL, intent(in):: VV(:, :) ! (KLON, KLEV) Merid wind at full levels

    ! 0.2 OUTPUTS
    REAL, intent(out):: zustr(:), zvstr(:) ! (KLON) Surface Stresses

    REAL, intent(inout):: d_u(:, :), d_v(:, :) 
    ! (KLON, KLEV) tendencies on winds

    ! O.3 INTERNAL ARRAYS
    REAL BVLOW(klon)

    INTEGER II, LL

    ! 0.3.0 TIME SCALE OF THE LIFE CYCLE OF THE WAVES PARAMETERIZED

    REAL DELTAT

    ! 0.3.1 GRAVITY-WAVES SPECIFICATIONS

    INTEGER, PARAMETER:: NK = 2, NP = 2, NO = 2, NW = NK * NP * NO
    INTEGER JK, JP, JO, JW
    REAL KMIN, KMAX ! Min and Max horizontal wavenumbers
    REAL CMIN, CMAX ! Min and Max absolute ph. vel.
    REAL CPHA ! absolute PHASE VELOCITY frequency
    REAL ZK(NW, KLON) ! Horizontal wavenumber amplitude
    REAL ZP(NW, KLON) ! Horizontal wavenumber angle 
    REAL ZO(NW, KLON) ! Absolute frequency !

    ! Waves Intr. freq. at the 1/2 lev surrounding the full level
    REAL ZOM(NW, KLON), ZOP(NW, KLON)

    ! Wave EP-fluxes at the 2 semi levels surrounding the full level
    REAL WWM(NW, KLON), WWP(NW, KLON)

    REAL RUW0(NW, KLON) ! Fluxes at launching level

    REAL RUWP(NW, KLON), RVWP(NW, KLON)
    ! Fluxes X and Y for each waves at 1/2 Levels

    INTEGER LAUNCH, LTROP ! Launching altitude and tropo altitude

    REAL XLAUNCH ! Controle the launching altitude
    REAL XTROP ! SORT of Tropopause altitude 
    REAL RUW(KLON, KLEV + 1) ! Flux x at semi levels
    REAL RVW(KLON, KLEV + 1) ! Flux y at semi levels

    REAL PRMAX ! Maximum value of PREC, and for which our linear formula
    ! for GWs parameterisation apply

    ! 0.3.2 PARAMETERS OF WAVES DISSIPATIONS

    REAL RDISS, ZOISEC ! COEFF DE DISSIPATION, SECURITY FOR INTRINSIC FREQ

    ! 0.3.3 BACKGROUND FLOW AT 1/2 LEVELS AND VERTICAL COORDINATE

    REAL H0 ! Characteristic Height of the atmosphere
    REAL DZ ! Characteristic depth of the source!
    REAL PR, TR ! Reference Pressure and Temperature

    REAL ZH(KLON, KLEV + 1) ! Log-pressure altitude

    REAL UH(KLON, KLEV + 1), VH(KLON, KLEV + 1) ! Winds at 1/2 levels
    REAL PH(KLON, KLEV + 1) ! Pressure at 1/2 levels
    REAL PSEC ! Security to avoid division by 0 pressure
    REAL PHM1(KLON, KLEV + 1) ! 1/Press at 1/2 levels
    REAL BV(KLON, KLEV + 1) ! Brunt Vaisala freq. (BVF) at 1/2 levels
    REAL BVSEC ! Security to avoid negative BVF

    !-----------------------------------------------------------------

    ! 1. INITIALISATIONS

    ! 1.1 Basic parameter

    ! Are provided from elsewhere (latent heat of vaporization, dry
    ! gaz constant for air, gravity constant, heat capacity of dry air
    ! at constant pressure, earth rotation rate, pi).

    ! 1.2 Tuning parameters of V14

    RDISS = 1. ! Diffusion parameter 

    PRMAX = 20. / 24. /3600.
    ! maximum of rain for which our theory applies (in kg/m^2/s)

    DZ = 1000. ! Characteristic depth of the source
    XLAUNCH=0.5 ! Parameter that control launching altitude
    XTROP=0.2 ! Parameter that control tropopause altitude
    DELTAT=24.*3600. ! Time scale of the waves (first introduced in 9b)

    KMIN = 2.E-5
    ! minimum horizontal wavenumber (inverse of the subgrid scale resolution)

    KMAX = 1.E-3 ! Max horizontal wavenumber
    CMIN = 1. ! Min phase velocity
    CMAX = 50. ! Max phase speed velocity

    TR = 240. ! Reference Temperature
    PR = 101300. ! Reference pressure
    H0 = RD * TR / RG ! Characteristic vertical scale height

    BVSEC = 5.E-3 ! Security to avoid negative BVF 
    PSEC = 1.E-6 ! Security to avoid division by 0 pressure
    ZOISEC = 1.E-6 ! Security FOR 0 INTRINSIC FREQ

    call assert(klon == (/size(pp, 1), size(tt, 1), size(uu, 1), &
         size(vv, 1), size(prec), size(zustr), size(zvstr), size(d_u, 1), &
         size(d_v, 1)/), "FLOTT_GWD_RANDO klon")
    call assert(klev == (/size(pp, 2), size(tt, 2), size(uu, 2), &
         size(vv, 2), size(d_u, 2), size(d_v, 2)/), "FLOTT_GWD_RANDO klev")

    IF(DELTAT < DTIME)THEN
       PRINT *, 'flott_gwd_rando: deltat < dtime!'
       STOP 1
    ENDIF

    IF (KLEV < NW) THEN
       PRINT *, 'flott_gwd_rando: you will have problem with random numbers'
       STOP 1
    ENDIF

    ! 2. EVALUATION OF THE BACKGROUND FLOW AT SEMI-LEVELS

    ! Pressure and Inv of pressure
    DO LL = 2, KLEV
       PH(:, LL) = EXP((LOG(PP(:, LL)) + LOG(PP(:, LL - 1))) / 2.)
       PHM1(:, LL) = 1. / PH(:, LL) 
    end DO

    PH(:, KLEV + 1) = 0. 
    PHM1(:, KLEV + 1) = 1. / PSEC
    PH(:, 1) = 2. * PP(:, 1) - PH(:, 2)

    ! Launching altitude

    LAUNCH=0
    LTROP =0
    DO LL = 1, KLEV
       IF (PH(KLON / 2, LL) / PH(KLON / 2, 1) > XLAUNCH) LAUNCH = LL
    ENDDO
    DO LL = 1, KLEV
       IF (PH(KLON / 2, LL) / PH(KLON / 2, 1) > XTROP) LTROP = LL
    ENDDO

    ! Log pressure vert. coordinate
    DO LL = 1, KLEV + 1 
       ZH(:, LL) = H0 * LOG(PR / (PH(:, LL) + PSEC))
    end DO

    ! BV frequency
    DO LL = 2, KLEV
       ! BVSEC: BV Frequency (UH USED IS AS A TEMPORARY ARRAY DOWN TO WINDS)
       UH(:, LL) = 0.5 * (TT(:, LL) + TT(:, LL - 1)) &
            * RD**2 / RCPD / H0**2 + (TT(:, LL) &
            - TT(:, LL - 1)) / (ZH(:, LL) - ZH(:, LL - 1)) * RD / H0
    end DO
    BVLOW = 0.5 * (TT(:, LTROP )+ TT(:, LAUNCH)) &
         * RD**2 / RCPD / H0**2 + (TT(:, LTROP ) &
         - TT(:, LAUNCH))/(ZH(:, LTROP )- ZH(:, LAUNCH)) * RD / H0

    UH(:, 1) = UH(:, 2)
    UH(:, KLEV + 1) = UH(:, KLEV)
    BV(:, 1) = UH(:, 2)
    BV(:, KLEV + 1) = UH(:, KLEV)
    ! SMOOTHING THE BV HELPS
    DO LL = 2, KLEV
       BV(:, LL)=(UH(:, LL+1)+2.*UH(:, LL)+UH(:, LL-1))/4.
    end DO

    BV=MAX(SQRT(MAX(BV, 0.)), BVSEC)
    BVLOW=MAX(SQRT(MAX(BVLOW, 0.)), BVSEC)

    ! WINDS
    DO LL = 2, KLEV
       UH(:, LL) = 0.5 * (UU(:, LL) + UU(:, LL - 1)) ! Zonal wind
       VH(:, LL) = 0.5 * (VV(:, LL) + VV(:, LL - 1)) ! Meridional wind
    end DO
    UH(:, 1) = 0.
    VH(:, 1) = 0.
    UH(:, KLEV + 1) = UU(:, KLEV)
    VH(:, KLEV + 1) = VV(:, KLEV)

    ! 3 WAVES CHARACTERISTICS CHOSEN RANDOMLY AT THE LAUNCH ALTITUDE

    ! The mod functions of weird arguments are used to produce the
    ! waves characteristics in an almost stochastic way

    JW = 0
    DO JP = 1, NP
       DO JK = 1, NK
          DO JO = 1, NO
             JW = JW + 1
             ! Angle
             DO II = 1, KLON
                ! Angle (0 or PI so far)
                ZP(JW, II) = (SIGN(1., 0.5 - MOD(TT(II, JW) * 10., 1.)) + 1.) &
                     * RPI / 2.
                ! Horizontal wavenumber amplitude
                ZK(JW, II) = KMIN + (KMAX - KMIN) * MOD(TT(II, JW) * 100., 1.)
                ! Horizontal phase speed
                CPHA = CMIN + (CMAX - CMIN) * MOD(TT(II, JW)**2, 1.)
                ! Absolute frequency is imposed
                ZO(JW, II) = CPHA * ZK(JW, II) 
                ! Intrinsic frequency is imposed
                ZO(JW, II) = ZO(JW, II) &
                     + ZK(JW, II) * COS(ZP(JW, II)) * UH(II, LAUNCH) &
                     + ZK(JW, II) * SIN(ZP(JW, II)) * VH(II, LAUNCH)
                ! Momentum flux at launch lev 
                RUW0(JW, II) = RUWMAX
             ENDDO
          end DO
       end DO
    end DO

    ! 4. COMPUTE THE FLUXES

    ! 4.1 Vertical velocity at launching altitude to ensure 
    ! the correct value to the imposed fluxes.

    DO JW = 1, NW

       ! Evaluate intrinsic frequency at launching altitude:
       ZOP(JW, :) = ZO(JW, :) &
            - ZK(JW, :) * COS(ZP(JW, :)) * UH(:, LAUNCH) &
            - ZK(JW, :) * SIN(ZP(JW, :)) * VH(:, LAUNCH) 

       ! VERSION WITH CONVECTIVE SOURCE

       ! Vertical velocity at launch level, value to ensure the
       ! imposed factor related to the convective forcing:
       ! precipitations.

       ! tanh limitation to values above prmax:
       WWP(JW, :) = RUW0(JW, :) &
            * (RD / RCPD / H0 * RLVTT * PRMAX * TANH(PREC / PRMAX))**2

       ! Factor related to the characteristics of the waves:
       WWP(JW, :) = WWP(JW, :) * ZK(JW, :)**3 / KMIN / BVLOW &
            / MAX(ABS(ZOP(JW, :)), ZOISEC)**3 

       ! Moderation by the depth of the source (dz here):
       WWP(JW, :) = WWP(JW, :) &
            * EXP(- BVLOW**2 / MAX(ABS(ZOP(JW, :)), ZOISEC)**2 * ZK(JW, :)**2 &
            * DZ**2)

       ! Put the stress in the right direction:
       RUWP(JW, :) = ZOP(JW, :) / MAX(ABS(ZOP(JW, :)), ZOISEC)**2 &
            * BV(:, LAUNCH) * COS(ZP(JW, :)) * WWP(JW, :)**2
       RVWP(JW, :) = ZOP(JW, :) / MAX(ABS(ZOP(JW, :)), ZOISEC)**2 &
            * BV(:, LAUNCH) * SIN(ZP(JW, :)) * WWP(JW, :)**2
    end DO

    ! 4.2 Uniform values below the launching altitude

    DO LL = 1, LAUNCH
       RUW(:, LL) = 0
       RVW(:, LL) = 0
       DO JW = 1, NW
          RUW(:, LL) = RUW(:, LL) + RUWP(JW, :)
          RVW(:, LL) = RVW(:, LL) + RVWP(JW, :)
       end DO
    end DO

    ! 4.3 Loop over altitudes, with passage from one level to the next
    ! done by i) conserving the EP flux, ii) dissipating a little,
    ! iii) testing critical levels, and vi) testing the breaking.

    DO LL = LAUNCH, KLEV - 1
       ! Warning: all the physics is here (passage from one level
       ! to the next)
       DO JW = 1, NW
          ZOM(JW, :) = ZOP(JW, :)
          WWM(JW, :) = WWP(JW, :)
          ! Intrinsic Frequency
          ZOP(JW, :) = ZO(JW, :) - ZK(JW, :) * COS(ZP(JW, :)) * UH(:, LL + 1) &
               - ZK(JW, :) * SIN(ZP(JW, :)) * VH(:, LL + 1) 

          ! No breaking (Eq.6)
          ! Dissipation (Eq. 8)
          WWP(JW, :) = WWM(JW, :) * EXP(- 2. * RDISS * PR / (PH(:, LL + 1) &
               + PH(:, LL)) * ((BV(:, LL + 1) + BV(:, LL)) / 2.)**3 &
               / MAX(ABS(ZOP(JW, :) + ZOM(JW, :)) / 2., ZOISEC)**4 &
               * ZK(JW, :)**3 * (ZH(:, LL + 1) - ZH(:, LL)))

          ! Critical levels (forced to zero if intrinsic frequency changes sign)
          ! Saturation (Eq. 12)
          WWP(JW, :) = min(WWP(JW, :), MAX(0., &
               SIGN(1., ZOP(JW, :) * ZOM(JW, :))) * ABS(ZOP(JW, :))**3 &
               / BV(:, LL + 1) * EXP(- ZH(:, LL + 1) / H0) * SAT**2 * KMIN**2 &
               / ZK(JW, :)**4)
       end DO

       ! Evaluate EP-flux from Eq. 7 and give the right orientation to
       ! the stress

       DO JW = 1, NW
          RUWP(JW, :) = SIGN(1., ZOP(JW, :))*COS(ZP(JW, :)) * WWP(JW, :)
          RVWP(JW, :) = SIGN(1., ZOP(JW, :))*SIN(ZP(JW, :)) * WWP(JW, :)
       end DO

       RUW(:, LL + 1) = 0.
       RVW(:, LL + 1) = 0.

       DO JW = 1, NW
          RUW(:, LL + 1) = RUW(:, LL + 1) + RUWP(JW, :) 
          RVW(:, LL + 1) = RVW(:, LL + 1) + RVWP(JW, :) 
       end DO
    end DO

    ! 5 CALCUL DES TENDANCES:

    ! 5.1 Rectification des flux au sommet et dans les basses couches

    RUW(:, KLEV + 1) = 0.
    RVW(:, KLEV + 1) = 0.
    RUW(:, 1) = RUW(:, LAUNCH)
    RVW(:, 1) = RVW(:, LAUNCH)
    DO LL = 1, LAUNCH
       RUW(:, LL) = RUW(:, LAUNCH+1)
       RVW(:, LL) = RVW(:, LAUNCH+1)
    end DO

    ! AR-1 RECURSIVE FORMULA (13) IN VERSION 4
    DO LL = 1, KLEV
       D_U(:, LL) = (1.-DTIME/DELTAT) * D_U(:, LL) + DTIME/DELTAT/REAL(NW) * &
            RG * (RUW(:, LL + 1) - RUW(:, LL)) &
            / (PH(:, LL + 1) - PH(:, LL)) * DTIME
       D_V(:, LL) = (1.-DTIME/DELTAT) * D_V(:, LL) + DTIME/DELTAT/REAL(NW) * &
            RG * (RVW(:, LL + 1) - RVW(:, LL)) &
            / (PH(:, LL + 1) - PH(:, LL)) * DTIME
    ENDDO

    ! Cosmetic: evaluation of the cumulated stress
    ZUSTR = 0.
    ZVSTR = 0.
    DO LL = 1, KLEV
       ZUSTR = ZUSTR + D_U(:, LL) / RG * (PH(:, LL + 1) - PH(:, LL))/DTIME
       ZVSTR = ZVSTR + D_V(:, LL) / RG * (PH(:, LL + 1) - PH(:, LL))/DTIME
    ENDDO

  END SUBROUTINE FLOTT_GWD_RANDO

end module FLOTT_GWD_rando_m
